/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: TCPConfigDlg.cpp 6810 2015-02-24 14:08:28Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2015-02-24 15:08:28 +0100 (Di, 24 Feb 2015) $
    $Revision: 6810 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
     Default Dialog

   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     3         08.08.11    RM       Updated dialog item enable/disable handling
     2         27.11.09    SS       Review
     1         28.07.09    SS       initial version
 
**************************************************************************************/

/*****************************************************************************/
/*! \file TCPConfigDlg.cpp                                                     
*   Implementation of the configuration dialog for the TCP connector         */
/*****************************************************************************/

#include "stdafx.h"
#include "TCPConfigDlg.h"
#include "netXConnectorErrors.h"
#include "Windowsx.h"
#include "commctrl.h"

#ifdef WIN64
  #include "resource_x64.h"
#else
  #include "resource_x86.h"
#endif

#include "TCPConfig.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_TCP netX TCP Connector                        */
/*! \{                                                                       */
/*****************************************************************************/

CTCPConfigDlg* CTCPConfigDlg::s_cInstance = 0; /*!< Pointer to single object instance */

extern HANDLE g_hModule; /*!< Global DLL module handle */

/////////////////////////////////////////////////////////////////////////////
/// Constructor
/////////////////////////////////////////////////////////////////////////////
CTCPConfigDlg::CTCPConfigDlg()
:m_hDialogWnd(NULL)
,m_cTCPConfig()
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CTCPConfigDlg::~CTCPConfigDlg() 
{
}

/////////////////////////////////////////////////////////////////////////////
/// Get object instance (singleton)
/////////////////////////////////////////////////////////////////////////////
CTCPConfigDlg* CTCPConfigDlg::GetInstance()
{
  if( !s_cInstance )
    s_cInstance = new CTCPConfigDlg();
  return s_cInstance;
}

/////////////////////////////////////////////////////////////////////////////
/// Create dialog window
///   \param hWndParent Parent window
///   \param szConfig   Configuration string
///   \return Window handle to created config dialog
/////////////////////////////////////////////////////////////////////////////
HWND CTCPConfigDlg::Create ( HWND hWndParent, const char* szConfig)
{
  m_cTCPConfig.Lock();

  /* Configuration string supplied ? */
  if (szConfig)
  {
    m_cTCPConfig.ParseString( szConfig);
  }

  m_hDialogWnd = CreateDialog((HINSTANCE)g_hModule, MAKEINTRESOURCE(IDD_DIALOG) , hWndParent, (DLGPROC)DialogProcStatic);
  
  /* Load and assign button icons */
  HANDLE hNew = LoadImage((HINSTANCE)g_hModule, MAKEINTRESOURCE(IDI_NEW)   , IMAGE_ICON,16,16,0);
  HANDLE hDel = LoadImage((HINSTANCE)g_hModule, MAKEINTRESOURCE(IDI_DELETE), IMAGE_ICON,16,16,0);
  SendMessage( GetDlgItem(m_hDialogWnd, IDC_BTN_DEL), BM_SETIMAGE, (LPARAM)IMAGE_ICON, (LPARAM)hDel);
  SendMessage( GetDlgItem(m_hDialogWnd, IDC_BTN_NEW), BM_SETIMAGE, (LPARAM)IMAGE_ICON, (LPARAM)hNew);

  /* Assign options to dialog items */
  PopulateComboBoxes();

  /* Initialize Up-Down Controls */
  InitUpDownControl();

  /* Display configuaration in dialog items */
  ShowSelInterfaceConfig();

  /* Set connector specific edit boxes */
  ShowLayerConfig();

  m_cTCPConfig.Unlock();

  return m_hDialogWnd;
}

/////////////////////////////////////////////////////////////////////////////
/// Destroy dialog window
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::Destroy( void)
{
  if (s_cInstance)
  {
    INT_PTR iRet = 0;
    EndDialog(m_hDialogWnd, iRet );
    delete s_cInstance;
    s_cInstance = NULL;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Get current dialog configuration as string
/// \return Configuration string
/////////////////////////////////////////////////////////////////////////////
long CTCPConfigDlg::GetConfiguration( CString& csRet)
{
  long lRet = NXCON_NO_ERROR;

  m_cTCPConfig.Lock();
  lRet = m_cTCPConfig.CreateString( csRet);
  m_cTCPConfig.Unlock();
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Set current dialog configuration as string
/// \param  szConfig Configuration string
/// \return NXCON_NO_ERROR if successful
/////////////////////////////////////////////////////////////////////////////
long CTCPConfigDlg::SetConfiguration( const char* szConfig)
{
  long lRet      = NXCON_NO_ERROR;

  m_cTCPConfig.Lock();
  if (NXCON_NO_ERROR == (lRet = m_cTCPConfig.ParseString( szConfig)))
  {
    HWND hIntfList = GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF);
    
    /* Remove all interfaces from combobox */
    ComboBox_ResetContent(hIntfList);

    /* Load interfaces to combobox */
    for (CTCPConfig::iterator iterIntf = m_cTCPConfig.begin(); iterIntf != m_cTCPConfig.end(); ++iterIntf) 
      ComboBox_AddString(hIntfList, *iterIntf);  
    ComboBox_SetCurSel(hIntfList, 0);

    ShowSelInterfaceConfig();
    ShowLayerConfig();
  }
  m_cTCPConfig.Unlock();
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Initiate Up/Down controls for edit boxes (Send/Reset/KeepAlive Timeout )
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::InitUpDownControl( void)
{
  HWND hItemWnd = NULL;
  long lMin     = 0;
  long lMax     = 0;

  /* Initialize up-down control for send timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_SEND);
  lMin = m_cTCPConfig.GetMinVal(eINTERFACE, NXCON_COMMON_SENDTIMEOUT);
  lMax = m_cTCPConfig.GetMaxVal(eINTERFACE, NXCON_COMMON_SENDTIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_SEND), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);

  /* Initialize up-down control for reset timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_RESET);
  lMin = m_cTCPConfig.GetMinVal(eINTERFACE, NXCON_COMMON_RESETTIMEOUT);
  lMax = m_cTCPConfig.GetMaxVal(eINTERFACE, NXCON_COMMON_RESETTIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_RESET), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);

  /* Initialize up-down control for keep alive timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_KEEPALIVE);
  lMin = m_cTCPConfig.GetMinVal(eINTERFACE, NXCON_COMMON_KEEPALIVETIMEOUT);
  lMax = m_cTCPConfig.GetMaxVal(eINTERFACE, NXCON_COMMON_KEEPALIVETIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_KEEPALIVE), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);
  
  /* Initialize up-down control for connect timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_CONNECT);
  lMin = m_cTCPConfig.GetMinVal(eLAYER, NXCON_TCPKEY_CONNECTTIMEOUT);
  lMax = m_cTCPConfig.GetMaxVal(eLAYER, NXCON_TCPKEY_CONNECTTIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_CONNECT), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);
}

/////////////////////////////////////////////////////////////////////////////
/// Populate Interface select Combobox
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::PopulateComboBoxes( void)
{
  /* Load interfaces to dialog controls */
  for (CTCPConfig::iterator iterIntf = m_cTCPConfig.begin(); iterIntf != m_cTCPConfig.end(); ++iterIntf) 
    ComboBox_AddString(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), *iterIntf);  
  ComboBox_SetCurSel(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), 0);  
}

/////////////////////////////////////////////////////////////////////////////
/// Set the layer dialog item sate (enable/disable)
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::SetLayerDlgItemState( void)
{
  if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE)))
  {
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_CONNECT), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_BTN_NEW), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_BTN_DEL), TRUE);

    /* Set the item states of the interface dialog */
    SetInterfaceDlgItemState();

  }else
  {
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_CONNECT), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_BTN_NEW), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_BTN_DEL), FALSE);

    /* Disable all interface elements from the sub window */
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS1), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_PORT), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_IPCOUNT), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), FALSE);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Set interface dialog item sate (enable/disable)
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::SetInterfaceDlgItemState( void)
{
  /* Read interface excluded from checkbox and store to config object */
  if (BST_CHECKED != Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE)))
  {
    ComboBox_Enable(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS1), TRUE);

    if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE)))
      EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), TRUE);
    
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_PORT), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_IPCOUNT), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), TRUE);

  }else
  {
    /* Disable all elements from the sub window */
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS1), FALSE);

    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), FALSE);

    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_PORT), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_IPCOUNT), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), FALSE);
  }
}


/////////////////////////////////////////////////////////////////////////////
/// Read interface configuration from dialog controls and store values 
/// in the layer and interface configuration Map
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::ReadFromDialogItems( NX_CONF_KEY_SCOPE_E eKeyScope, CONFIGKEY_MAP& cmKeyMap)
{
  char szValue[MAX_PATH] = "";  
  
  switch (eKeyScope)
  {
    case eINTERFACE:
      /* Read keep alive timeout value from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_COMMON_KEEPALIVETIMEOUT, szValue));

      /* Read send timeout value from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_COMMON_SENDTIMEOUT, szValue));

      /* Read reset timeout value from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_COMMON_RESETTIMEOUT, szValue));

      /* Read TCP port from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_PORT), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_TCPKEY_PORT, szValue));

      /* Read ip address (begin of range) from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS1), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_TCPKEY_IPBEGIN, szValue));

      if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE)))
      {
        /* Read ip address (end of range) if checkbox is checked and store to config object */
        Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), szValue, sizeof(szValue)/sizeof(*szValue));

        /* If ip address is cleared copy begin of range ip address */
        if (0 == inet_addr(szValue))
        {
          Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS1), szValue, sizeof(szValue)/sizeof(*szValue));
          Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), szValue);
        }
        cmKeyMap.insert(make_pair(NXCON_TCPKEY_IPEND, szValue));
        EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), TRUE);
      } else
      {
        /* Clear ip address range if checkbox is unchecked and disable editbox */
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), "0.0.0.0");
        cmKeyMap.insert(make_pair(NXCON_TCPKEY_IPEND, "0.0.0.0"));
        EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), FALSE);
      }

      /* Read interface excluded from checkbox and store to config object */
      if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE)))
      {
        cmKeyMap.insert(make_pair(NXCON_COMMON_EXCLUDE, "1"));
      }else
      {
        cmKeyMap.insert(make_pair(NXCON_COMMON_EXCLUDE, "0"));
      }
      
      break;
      
    case eLAYER:
      /* Read connection timeout from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_CONNECT), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_TCPKEY_CONNECTTIMEOUT, szValue));

      /* Read connector enabled from checkbox and store to config object */
      if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE)))
      {
        cmKeyMap.insert(make_pair(NXCON_COMMON_ENABLED, "1"));

      }else
      {
        cmKeyMap.insert(make_pair(NXCON_COMMON_ENABLED, "0"));
      }

      break;

    default:
      return;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Write interface configuration from the interface map to the dialog controls
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::WriteToDialogItems( NX_CONF_KEY_SCOPE_E eKeyScope, CONFIGKEY_MAP& cmKeyMap)
{
  CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.end();

  switch (eKeyScope)
  {
    case eINTERFACE:
      /* Read keep alive timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_KEEPALIVETIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), iterKeys->second);

      /* Read send timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_SENDTIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), iterKeys->second);

      /* Read reset timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_RESETTIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), iterKeys->second);

      /* Read TCP port from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_TCPKEY_PORT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_PORT), iterKeys->second);

      /* Read ip address (begin of range) from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_TCPKEY_IPBEGIN)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS1), iterKeys->second);

      /* Read ip address (end of range) from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_TCPKEY_IPEND)))
      {
        if (0 != inet_addr(iterKeys->second))
        {
          /* End of range ip address is valid, check checkbox and write ip address to editbox */
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE), BST_CHECKED);
          EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), TRUE);
          Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), iterKeys->second);
        } else
        {
          /* End of range ip address is cleared, uncheck checkbox and disable editbox */
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_IPRANGE), BST_UNCHECKED);
          EnableWindow(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), FALSE);
          Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_IPADDRESS2), iterKeys->second);
        }
      }

      /* Read interface exclude value from config object and write to checkbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_EXCLUDE)))
      {
        if (atoi(iterKeys->second))
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), BST_CHECKED);
         else
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), BST_UNCHECKED);
      }      
      break;
      
    case eLAYER:
      /* Read connect timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_TCPKEY_CONNECTTIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_CONNECT), iterKeys->second);
      
      /* Read connector enabled value from config object and write to checkbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_ENABLED)))
      {
        if (atoi(iterKeys->second))
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE), BST_CHECKED);
         else
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE), BST_UNCHECKED);
      }
      break;
    default:
      return;
  }
}

///////////////////////////////////////////////////////////////////////////
/// Read interface configuration from dialog controls and store values 
/// in the layer and interface configuration Map
///////////////////////////////////////////////////////////////////////////
CString CTCPConfigDlg::GetSelInterfaceName( void)
{
  char szName[MAX_PATH] = "";
  ComboBox_GetText(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), szName, sizeof(szName)/sizeof(*szName)); 
  return CString(szName);
}

/////////////////////////////////////////////////////////////////////////////
/// Show Configuration of selected interface in the dialog
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::ShowSelInterfaceConfig( void)
{
  CONFIGKEY_MAP cmKeyMap;  
  CString csIntfName = GetSelInterfaceName();

  if ( NXCON_NO_ERROR == m_cTCPConfig.LoadInterfaceConfig( csIntfName, cmKeyMap))
  {
    /* Interface configuration available, so enable controls for interface configuration */
    WriteToDialogItems(eINTERFACE, cmKeyMap);
    SetIPAdressCounter( csIntfName);
  }
  
  /* Setup item states */
  SetInterfaceDlgItemState();
 
}

/////////////////////////////////////////////////////////////////////////////
/// Show Configuration of selected interface in the dialog
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::ShowLayerConfig( void)
{
  CONFIGKEY_MAP cmKeyMap;
  if (NXCON_NO_ERROR == m_cTCPConfig.LoadLayerConfig( cmKeyMap))
    WriteToDialogItems(eLAYER, cmKeyMap);

  SetLayerDlgItemState();
}

/////////////////////////////////////////////////////////////////////////////
/// Set IP address counter 
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::SetIPAdressCounter( const char* szIntfName)
{
  CTCPConfig::iterator iterIntf = m_cTCPConfig.find( szIntfName);
  if (m_cTCPConfig.end() != iterIntf)
  {
    CString csIPSize;
    csIPSize.AppendFormat("%lu", iterIntf.GetIPRangeSize());
    Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_IPCOUNT), csIPSize);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Add interface to the dialog controls and to the interface map
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::OnBtnClickedNew( void)
{
  HWND          hIntfList = GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF);
  CString       csIntfName;

  m_cTCPConfig.Lock();

  csIntfName = m_cTCPConfig.CreateInterfaceName();

  if ( NXCON_NO_ERROR == m_cTCPConfig.CreateInterfaceConfig(csIntfName))
  {
    ComboBox_AddString(hIntfList, csIntfName);    
    ComboBox_SetCurSel(hIntfList, ComboBox_FindStringExact(hIntfList, -1, csIntfName));
    ShowSelInterfaceConfig();
  }
  
  m_cTCPConfig.Unlock();

}

/////////////////////////////////////////////////////////////////////////////
/// Remove interface from the dialog controls and from the interface map
/////////////////////////////////////////////////////////////////////////////
void CTCPConfigDlg::OnBtnClickedRemove( void)
{
  HWND hIntfList = GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF);
  int  iListIdx  = 0;
  
  m_cTCPConfig.Lock();

  if ( (CB_ERR != (iListIdx = ComboBox_GetCurSel(hIntfList))) &&
       (NXCON_NO_ERROR == m_cTCPConfig.RemoveInterfaceConfig( GetSelInterfaceName())))
  {
    ComboBox_DeleteString(hIntfList, iListIdx);
    ComboBox_SetCurSel(hIntfList, 0);
    ShowSelInterfaceConfig();
  }
  
  m_cTCPConfig.Unlock();

}

/////////////////////////////////////////////////////////////////////////////
/// Function to handle dialog actions
/// \param hDlg    Window handle of the dialog
/// \param message Window messages
/// \param wParam  Parameters associated with window message
/// \param lParam  Parameters associated with window message
/// \return TRUE if message is processed
/////////////////////////////////////////////////////////////////////////////
INT_PTR CALLBACK CTCPConfigDlg::DialogProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  UNREFERENCED_PARAMETER(lParam);

  switch (message)
  {
    case WM_INITDIALOG:
      return TRUE;

    case WM_COMMAND:
      switch (LOWORD(wParam))
      {
        case IDC_BTN_NEW:
          /* 'New Interface' button pushed */
          OnBtnClickedNew();
          return TRUE;

        case IDC_BTN_DEL:
          /* 'Del Interface' button pushed */
          OnBtnClickedRemove();
          return TRUE;

        case IDC_COMBO_INTF:
          /* Changed interface in list */
          if (CBN_SELCHANGE == HIWORD(wParam)) 
          { 
            m_cTCPConfig.Lock();
            ShowSelInterfaceConfig();
            m_cTCPConfig.Unlock();
          }
          return TRUE;

        case IDC_CHECK_ENABLE:
          if (BN_CLICKED == (HIWORD(wParam))) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            ReadFromDialogItems( eLAYER, cmKeyMap);
            SetLayerDlgItemState();
            m_cTCPConfig.Lock();
            m_cTCPConfig.StoreLayerConfig(cmKeyMap);
            m_cTCPConfig.Unlock();
          }
          return TRUE;
        
        case IDC_CHECK_IPRANGE:
        case IDC_CHECK_EXCLUDE:
          if (BN_CLICKED == (HIWORD(wParam))) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            ReadFromDialogItems( eINTERFACE, cmKeyMap);
            SetInterfaceDlgItemState();
            CString csName = GetSelInterfaceName();
            m_cTCPConfig.Lock();
            m_cTCPConfig.StoreInterfaceConfig(csName, cmKeyMap);
            m_cTCPConfig.Unlock();
            SetIPAdressCounter( csName);
          }
          return TRUE;

        case IDC_IPADDRESS1:
        case IDC_IPADDRESS2:
        case IDC_EDIT_KEEPALIVE:
        case IDC_EDIT_SEND:
        case IDC_EDIT_RESET:
        case IDC_EDIT_PORT:
          if (EN_KILLFOCUS == HIWORD(wParam)) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            CString csName = GetSelInterfaceName();
            ReadFromDialogItems( eINTERFACE, cmKeyMap);
            SetInterfaceDlgItemState();
            m_cTCPConfig.Lock();
            if (NXCON_NO_ERROR != m_cTCPConfig.StoreInterfaceConfig( csName, cmKeyMap))
              Edit_Undo(GetDlgItem(m_hDialogWnd, LOWORD(wParam)));
            m_cTCPConfig.Unlock();            
            SetIPAdressCounter( csName);
          }
          return TRUE;

        case IDC_EDIT_CONNECT:
          if (EN_KILLFOCUS == HIWORD(wParam)) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            ReadFromDialogItems( eLAYER, cmKeyMap);
            SetLayerDlgItemState();
            m_cTCPConfig.Lock();
            if (NXCON_NO_ERROR != m_cTCPConfig.StoreLayerConfig( cmKeyMap))
              Edit_Undo(GetDlgItem(m_hDialogWnd, LOWORD(wParam)));      
            m_cTCPConfig.Unlock();
          }

      }
    case WM_HSCROLL:
        return 0;

    case WM_CLOSE:
        EndDialog(hDlg, TRUE);
        return TRUE;
  }
  
  //return ::DefWindowProc( hDlg, message, wParam, lParam);
  return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
/// Wrapper function to handle dialog actions
/// \param hDlg    Window handle of the dialog
/// \param message Window messages
/// \param wParam  Parameters associated with window message
/// \param lParam  Parameters associated with window message
/// \return TRUE if message is processed
/////////////////////////////////////////////////////////////////////////////
INT_PTR CALLBACK CTCPConfigDlg::DialogProcStatic(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  return(s_cInstance->DialogProc(hDlg, message, wParam, lParam));
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/